{
 Author:    Craig Ward,
            H Van Tasell, 76602.1123@compuserve.com

 Copyright: none - public domain

 Date:      18/6/96

 Version:   1.33


 Overview:  The TcwXTab component is a component, descended from TStringGrid, which
            allows cross-tabulations of a table's data to be built.


 XTab's:    A crosstab is the representation of de-normalised table-data. Typically
            you'd have a summary by the unique values contained in a row field and
            a column field.

            In this component, you can have multiple results in a cell (a max of 3), each performing
            a discrete mathematical operation, and displaying this data in the format
            that you specify.


 Calling:   The method Execute will execute the crosstab (returns true if successful)
            The method SaveXTab will save the crosstab as a Paradox table (returns true if successful)


 Update:    The following amendments have been made:

             [0] resolution for bug in reading negative currency values.

             [1] the OnDrawCell routine now ignores row\column headings since I felt it looked
                 aesthetically unattractive to have text with hard-returns in the headings...

             [2] the crosstab can now have summary fields for both rows and columns (this is
                 the AggRowCols property). Note that the summary cells used a maroon font, and
                 in the case of using the SaveXTab procedure, they are not saved to the result
                 table (since that would *seriously* violate normalisation rules).

             [3] the crosstab now has an additional property which allows the cells to resize
                 to the maximum text displayed (this is the AutoResize property). In this case,
                 the component will make row-heights big enough to fit the lines in a cell,
                 whilst the column-widths will be made big enough to fit the maximum size
                 of text in each respective column heading.

             [4] the crosstab now uses pointers to the custom types used in the PopXTab routine.

             [5] a new property, "EmptyCellChar", allows the developer to choose between blanks
                 or zeros for empty cells.

             [6] a new property, "DataIsCurrent", is a bool that when true means that the
                 data contained within the crosstab is up-to-date. It should only be
                 used in master-detail relationships.

                 It is set to TRUE on execution, and must be manually set to
                 false (the best place to do this is on the master table's
                 datasource's OnDataChange event).

             [7] The component has been amended so that it now works fine with extensions
                 on table-names, and spaces in field-names.


 Bugs:      The following undocumented features are known:

             [1] Though the editor will allow you to select calculated fields, the crosstab
                 will actually run into GPF if you try to run it where either the row or
                 column fields are calculated.
             [2] The auto-resize function works on the text in column headings, and not
                 on the text in cells. Therefore, a column could be resized so that the
                 heading fits, but the value in the column's cells is not properly visible.


 VCL:       IMHO, this component is okay. I've seen better (one that springs to mind is
            the TCrosstab component by Kevin Liu - INTERNET:kliu@oodb.syscom.com.tw -
            which is particularly impressive in that you can set how many columns and
            rows to summarise by). However, I don't know of any crosstab components where
            they are free, and the source is available at no charge.


 Thanks:    Thanks goto the following:

             [1] Dennis Passmore (71640.2464@compuserve.com) for help in drawing multi-line
                 text in a cell
             [2] Greg Tresdell (74131.2175@compuserve.com) for help in writing custom
                 property editors. Check out his "Gray Paper" for more information...
             [3] The following for help in getting text metrics:
                  Pat Ritchey (700007.4660@compuserve.com)
                  Julian Bucknall (72662.1324@compuserve.com)
                  Harley L Pebley (103330.2334@compuserve.com)
             [4] Harry Van Tasell for taking time to add improvements to the component
             [5] Joe Griffin (100745.1251@compuser.com) for pointing a couple
                 of bugs in the component.

**********************************************************************************}
unit Cwxtab;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  DsgnIntf, Forms, Dialogs, Grids, DB, DBTables;


{constants for array sizes}
const
 iSizeOfArray = 2;


type

  {custom type - possible math operations to be carried out}
  TMathOp = (sum, avg, min, max, count);

  {custom type - empty character}
  TEmptyChar = (ecBlank,ecZero);

  {custom type - possible display options for results}
  TResultFormat = (currency_format, integer_format, real_format);

  {custom type - returns cell contents}
  TarrFloat = array[0..iSizeOfArray] of extended;
  parrFloat = ^TarrFloat;

  {custom type - record which stores the data in the TXObject}
  TSummaryField = record
   FieldName: string;
   MathOp: TMathOp;
   Format: TResultFormat;
  end;
  {custom type - array of the previous custom type}
  TarrSummaryField = array[0..iSizeOfArray] of TSummaryField;
  parrSummaryField = ^TarrSummaryField;

  {custom type - record which stores data for the current cell}
  TCellData = record
   Result: extended;
   Min: extended;
   Max: extended;
   Count: integer;
   Avg: extended;
   Sum: extended;
  end;
  {custom type - array of the previous custom type}
  TarrCellData = array[0..iSizeOfArray] of TCellData;
  parrCellData = ^TarrCellData;


  {custom type - this is the object that the property editor is editing}
  TXObject = class(TPersistent)
  private
    FRowField: string;
    FColField: string;
    FSumField1: string;
    FSumField2: string;
    FSumField3: string;
    FMathField1: TMathOp;
    FMathField2: TMathOp;
    FMathField3: TMathOp;
    FFormatField1: TResultFormat;
    FFormatField2: TResultFormat;
    FFormatField3: TResultFormat;
    FOnChange: TNotifyEvent;
    FTable: TTable;
    procedure SetRowField(const value: string);
    procedure SetColField(const value: string);
    procedure SetSumField1(const value: string);
    procedure SetSumField2(const value: string);
    procedure SetSumField3(const value: string);
    procedure SetTable(value: TTable);
    procedure SetSumField1MathOp(const value: TMathOp);
    procedure SetSumField2MathOp(const value: TMathOp);
    procedure SetSumField3MathOp(const value: TMathOp);
    procedure SetSumField1Format(const value: TResultFormat);
    procedure SetSumField2Format(const value: TResultFormat);
    procedure SetSumField3Format(const value: TResultFormat);
  public
    procedure Changed;
  published
    property RowField: string read FRowField write SetRowField;
    property ColumnField: string read FColField write SetColField;
    property SummaryField1: string read FSumField1 write SetSumField1;
    property SummaryField2: string read FSumField2 write SetSumField2;
    property SummaryField3: string read FSumField3 write SetSumField3;
    property SumField1MathOp: TMathOp read FMathField1 write SetSumField1MathOp;
    property SumField2MathOp: TMathOp read FMathField2 write SetSumField2MathOp;
    property SumField3MathOp: TMathOp read FMathField3 write SetSumField3MathOp;
    property SumField1Format: TResultFormat read FFormatField1 write SetSumField1Format;
    property SumField2Format: TResultFormat read FFormatField2 write SetSumField2Format;
    property SumField3Format: TResultFormat read FFormatField3 write SetSumField3Format;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property Table: TTable read FTable write SetTable;
  end;


  {the declaration for the crosstab component}
  TcwXTab = class(TStringGrid)
  private
    { Private declarations }
    Query1: TQuery;
    FXTab: TXObject;
    FTable: TTable;
    FAgg: boolean;
    FAutoResizeCells: boolean;
    FEmptyChar: TEmptyChar;
    FCurrent: boolean;
    procedure SetCurrent(value: boolean);
    procedure SetEmptyChar(value: TEmptyChar);
    procedure SetTable(value: TTable);
    function PopRows: boolean;
    function PopCols: boolean;
    function PopXTab: boolean;
    function AddSummaries: boolean;
    function RunQuery(const sSQL: string): boolean;
    procedure SetXtab(value: TXObject);
    procedure SetAgg(value: boolean);
    procedure SetCells(value: boolean);
    function GetCellContents(const iRow, iCol: integer): TarrFloat;
    procedure DrawCellContents(const iRow, iCol: integer; eArray: TarrFloat);
    function FindBiggestStr(const sCell: string): integer;
  protected
    { Protected declarations }
    procedure DrawCell(Sender: TObject; Col, Row: Longint;Rect: TRect; State: TGridDrawState);
  public
    { Public declarations }
    function UpdateXTab: boolean;
    function Execute: boolean;
    function SaveXTab(const sTable: string): boolean;
    constructor create(aOwner: TComponent); override;
    destructor destroy; override;
  published
    { Published declarations }
    property DataIsCurrent: boolean read FCurrent write SetCurrent;
    property Crosstab: TXObject read FXTab write SetXTab;
    property Table: TTable read FTable write SetTable;
    property AggRowsCols: boolean read FAgg write SetAgg;
    property AutoResize: boolean read FAutoResizeCells write SetCells;
    property EmptyCellChar: TEmptyChar read FEmptyChar write SetEmptyChar;
  end;


procedure Register;

{variables}
var
 lparrSumFields: parrSummaryField; {stores details for summary fields}
 iInc: integer;               {common integer var}



implementation


uses
 cwXTEdit;


{****VCL preferences************************************************************}

{register}
procedure Register;
begin
  RegisterComponents('cw_apps', [TcwXTab]);
  RegisterComponentEditor(TcwXTab, TXTabEditor);
  RegisterPropertyEditor(TypeInfo(TXObject),nil,'',TXTabProperty);
end;


{constructor}
constructor TcwXTab.create(aOwner: TComponent);
begin
 inherited create(aOwner);

  Query1 := TQuery.create(self);
  FXTab := TXObject.create;

  {set defaults for cwXTab}
  options := options + [goColSizing, goRowSizing, goColMoving, goRowMoving];
  OnDrawCell := drawCell;
  EmptyCellChar := ecBlank;

end;


{destructor}
destructor TcwXTab.destroy;
begin
 Query1.free;
 FXTab.free;
 inherited destroy;
end;


{set current}
procedure TcwXTab.SetCurrent(value: boolean);
begin
 if value <> FCurrent then
  begin
   FCurrent := value;
  end;
end;


{set empty char}
procedure TcwXTab.SetEmptyChar(value: TEmptyChar);
begin
 if value <> FEmptyChar then
  begin
   FEmptyChar := value;
  end;
end;


{set custom object which holds crosstab}
procedure TcwXTab.SetXTab(value : TXObject);
begin
 FXTab := (value);
end;

{set Table}
procedure TcwXTab.SetTable(value: TTable);
begin
 if value <> FTable then
  begin
   FTable := Value;
  end;
 {set object's table to match}
 crosstab.Table := value;
end;


{set aggregate}
procedure TcwXTab.SetAgg(value: boolean);
begin
 if value <> FAgg then
  FAgg := value;
end;

{set cell heights and widths}
procedure TcwXTab.SetCells(value: boolean);
begin
 if value <> FAutoResizeCells then
  FAutoResizeCells := value;
end;

{***custom object preferences**************************************************}

{set field}
procedure TXObject.SetSumField1Format(const value: TResultFormat);
begin
 if value <> FFormatField1 then
  FFormatField1 := value;
end;

{set field}
procedure TXObject.SetSumField2Format(const value: TResultFormat);
begin
 if value <> FFormatField2 then
 FFormatField2 := value;
end;

{set field}
procedure TXObject.SetSumField3Format(const value: TResultFormat);
begin
if value <> FFormatField3 then
 FFormatField3 := value;
end;

{set field}
procedure TXObject.SetSumField1MathOp(const value: TMathOp);
begin
if value <> FMathField1 then
 FMathField1 := value;
end;

{set field}
procedure TXObject.SetSumField2MathOp(const value: TMathOp);
begin
if value <> FMathField2 then
 FMathField2 := value;
end;

{set field}
procedure TXObject.SetSumField3MathOp(const value: TMathOp);
begin
if value <> FMathField3 then
 FMathField3 := value;
end;

{set field}
procedure TXobject.SetRowField(const value: string);
begin
if value <> FRowField then
 FRowField := value;
end;

{set field}
procedure TXobject.SetColField(const value: string);
begin
if value <> FColField then
 FColField := value;
end;


{set changed}
procedure TXobject.Changed;
begin
 if Assigned(FOnChange) then FOnChange(Self);
end;

{set object's table reference}
procedure TXObject.SetTable(value: TTable);
begin
if value <> FTable then
 FTable := value;
end;

{set field}
procedure TXObject.SetSumField1(const value: string);
begin
if value <> FSumField1 then
 FSumField1 := value;
end;

{set field}
procedure TXObject.SetSumField2(const value: string);
begin
if value <> FSumField2 then
 FSumField2 := value;
end;

{set field}
procedure TXObject.SetSumField3(const value: string);
begin
if value <> FSumField3 then
 FSumField3 := value;
end;




{***XTab custom routines*******************************************************}

{execute}
function TcwXTab.Execute: boolean;
begin
 result := false;
 if Table = nil then exit;

 screen.cursor := crHourGlass;
 New(lparrSumFields);
 try
  {initialise array}
  lparrSumFields^[0].FieldName := Crosstab.SummaryField1;
  lparrSumFields^[0].MathOp := Crosstab.SumField1MathOp;
  lparrSumFields^[0].Format := Crosstab.SumField1Format;
  {***}
  lparrSumFields^[1].FieldName := Crosstab.SummaryField2;
  lparrSumFields^[1].MathOp := Crosstab.SumField2MathOp;
  lparrSumFields^[1].Format := Crosstab.SumField2Format;
  {***}
  lparrSumFields^[2].FieldName := Crosstab.SummaryField3;
  lparrSumFields^[2].MathOp := Crosstab.SumField3MathOp;
  lparrSumFields^[2].Format := Crosstab.SumField3Format;

  {execute methods to populate the crosstab}
  if popRows then
   if popCols then
    if popXTab then
     begin
      {auto-resize}
      if AutoResize then
       if not UpdateXTab then
        messageDlg('Error in resizing crosstab!',mtWarning,[mbOK],0);
      {aggregates}
      if AggRowsCols then
       if not AddSummaries then
        messageDlg('Error in adding summaries to crosstab!',mtWarning,[mbOK],0);
      result := true;
     end;

 {cleanup}
 finally
  Dispose(lparrSumFields);
  FCurrent := true;
  screen.cursor := crDefault;
 end;

end;


{update XTAb}
function TcwXTab.UpdateXTab: boolean;
var
 FontMetrics: TTextMetric;
 w, wAgg: word;
begin
 result := false;
 screen.cursor := crHourGlass;

 {initialise}
 w := 0;
 wAgg := 1;

 {set row heights}
 self.RowHeights[0] := 24;                             {set first row height to default}
 for iInc := 0 to iSizeOfArray do
  if lparrSumFields^[iInc].FieldName <> '' then inc(w,1);   {find how many lines of text in cells}
 GetTextMetrics(Canvas.Handle, FontMetrics);           {get current text-metrics}
 for iInc := 1 to (self.RowCount - 1) do               {apply height to rows}
  self.RowHeights[iInc] := (FontMetrics.tmHeight + FontMetrics.tmInternalLeading) * w;

 {set column width for row-headers}
 w := FindBiggestStr( self.cells[0,1] );               {get initial value}
 for iInc := 1 to (self.rowCount - 1) do
  begin
   if w < FindBiggestStr( self.cells[0,iInc] ) then
    w := FindBiggestStr( self.cells[0,iInc] );
  end;
 if w < 64 then w := 64;    {set min width}
 self.colWidths[0] := w;

 {set column widths for columns}
 if AggRowsCols then inc(wAgg,1);
 for iInc := 1 to (self.colCount - wAgg) do
  begin
   w := FindBiggestStr( self.cells[iInc,0] );
   if w < 64 then w := 64;  {set min width}
   self.ColWidths[iInc] := w;
  end;
  self.ColWidths[iInc] := FindBiggestStr( self.cells[iInc,0] );
 {use default for last column}
 if AggRowsCols then self.colWidths[self.colCount -1] := 64;

 result := true;
 screen.cursor := crDefault;

end;


{return required pixels for string passed}
function TcwXTab.FindBiggestStr(const sCell: string): integer;
begin
 result := canvas.textWidth(sCell) + 3; {return size of string passed in pixels, plus a value for grid spacing}
end;


{run query - this is used by PopRows and PopCols to execute their SQL statements}
function TcwXTab.RunQuery(const sSQL: string): boolean;
begin
 result := false;
 Query1.active := false;
 try
  Query1.DatabaseName := Table.DatabaseName;
  Query1.SQL.clear;
  Query1.SQL.Add(sSQL);
  Query1.active := true;
  result := true;
 except on EDatabaseError do
  begin
   cursor := crDefault;
   messageDlg('Error in execution of query!',mtWarning,[mbOK],0);
  end;
 end;
end;


{populate rows - read distinct values from row field into the grid's rows}
function TcwXTab.PopRows: boolean;
var
 i, iAdd: integer;
begin
 result := false;
 {if row aggregate then add two additional rows as opposed to one}
 if AggRowsCols then
  iAdd := 2
 else
  iAdd := 1;
 if RunQuery('select distinct '+ChangeFileExt(Table.TableName,'')+'."'+
             Crosstab.RowField+'" from '+ChangeFileExt(Table.TableName,'')) then
  begin
   Query1.first;
   i := 1;
   self.rowCount := Query1.recordCount + iAdd;
   while not query1.eof do
    begin
     self.cells[0,i] := Query1.fields[0].text;
     inc(i);
     Query1.next;
    end;
   {add summary label}
   if AggRowsCols then self.cells[0,(self.rowCount -1)] := 'Total';
   result := true;
  end;
end;


{populate columns - read distinct values from column field into the grid's columns}
function TcwXTab.PopCols: boolean;
var
 i, iAdd: integer;
begin
 result := false;
 {if row aggregate then add two additional rows as opposed to one}
 if AggRowsCols then
  iAdd := 2
 else
  iAdd := 1;
 if RunQuery('select distinct '+ChangeFileExt(Table.TableName,'')+'."'+
             Crosstab.ColumnField+'" from '+ChangeFileExt(Table.TableName,'')) then
  begin
   Query1.first;
   i := 1;
   self.ColCount := Query1.recordCount + iAdd;
   while not Query1.eof do
    begin
     self.cells[i,0] := Query1.fields[0].text;
     inc(i);
     Query1.next;
    end;
   {add summary label}
   if AggRowsCols then self.cells[(self.ColCount -1),0] := 'Total';
   result := true;
  end;
end;


{populate cross-tab - the crux of the component. Lots of code that basically concerns itself with
 iterating through the table, finding values in the row and column fields that match the current
 cell row\column headings, and then performing the math operation as specified by the developer}
function TcwXTab.PopXTab: boolean;
var
 eSum, eMin, eMax: extended;
 iRow, iCol, iFldCol, iFldRow: integer;
 eCell: parrCellData;
 e: parrFloat;
 bFindMinOrMax: boolean;
 wRow, wCol: word;
begin

 result := false;
 New(e);
 New(eCell);

 try

  {find index in table of row and column fields}
  iFldCol := Table.FieldbyName( Crosstab.ColumnField ).Index;
  iFldRow := Table.FieldbyName( Crosstab.RowField ).Index;

  {iterate by row}
  wRow := self.RowCount;
  if AggRowsCols then dec(wRow,1);
  for iRow := 1 to wRow do
   begin

    {iterate by row, column}
    wCol := self.colCount;
    if AggRowsCols then dec(wCol,1);
    for iCol := 1 to wCol do
     begin

      {clean out cell array}
      for iInc := 0 to iSizeOfArray do
       begin
        eCell^[iInc].Result := 0;
        eCell^[iInc].Min := 0;
        eCell^[iInc].Max := 0;
        eCell^[iInc].Count := 0;
        eCell^[iInc].Avg := 0;
        eCell^[iInc].Sum := 0;
       end;

      {check to see if developer has specified a max or min operation. We check for this, since either of these
       will require an iteration through the table, per field defined, in order to find an initial value
       for each cell\row which serve as a starting point for both min\max. Obviously, if the developer hasn't
       specified either of these operations then it's an un-neccessary delay that we can avoid}
      for iInc := 0 to iSizeOfArray do
       if (lparrSumFields^[iInc].MathOp = min) or (lparrSumFields^[iInc].MathOp = max) then
        bFindMinOrMax := true;

      {developer needs min\max value}
      if bFindMinOrMax then
       for iInc := 0 to iSizeOfArray do
        if (lparrSumFields^[iInc].FieldName <> '')
         and ( (lparrSumFields^[iInc].MathOp = min) or (lparrSumFields^[iInc].MathOp = max) ) then
          begin
           table.First;
           {iterate through table}
           while not table.eof do
            begin
             {get first value in field, where the cell's row\column headings
              are equal to the row\column fields}
             if (CompareText(Table.fields[iFldCol].Text,self.cells[iCol,0]) = 0)
              and (CompareText(Table.fields[iFldRow].Text,self.cells[0,iRow]) = 0) then
               begin
                eCell^[iInc].Min := StrToFloat(Table.fieldByName( lparrSumFields^[iInc].FieldName ).text);
                eCell^[iInc].Max := StrToFloat(Table.fieldByName( lparrSumFields^[iInc].FieldName ).text);
                break; {break from loop since we only need first value}
               end;
             table.next;
            end;
          end;

      {this is the start of the main crosstab routine...we now iterate through table, by values that
       match row and column, performing the developer's math operation}
      Table.first;
      while not Table.eof do
       begin
        if (CompareText(Table.fields[iFldCol].Text,self.cells[iCol,0]) = 0)
         and (CompareText(Table.fields[iFldRow].Text,self.cells[0,iRow]) = 0) then
          begin

           {add to cell array the summary values}
           for iInc := 0 to iSizeOfArray do
            begin

             {switch on operation}
             if lparrSumFields^[iInc].FieldName <> '' then
              case lparrSumFields^[iInc].MathOp of
               sum:               {sum up values}
                begin
                 eCell^[iInc].Result :=
                 eCell^[iInc].Result + StrToFloat(Table.fieldByName( lparrSumFields^[iInc].FieldName ).text);
                end;
               count:             {count}
                begin
                 inc(eCell^[iInc].Count,1);
                 eCell^[iInc].Result := eCell^[iInc].Count;
                end;
               avg:               {average}
                begin
                 inc(eCell^[iInc].Count,1);
                 eCell^[iInc].Sum := eCell^[iInc].Sum +
                  StrToFloat(Table.fieldByName( lparrSumFields^[iInc].FieldName ).text);
                 eCell^[iInc].Result := eCell^[iInc].Sum / eCell^[iInc].Count;
                end;
               min:                {minimum}
                begin
                 if StrToFloat(Table.fieldByName( lparrSumFields^[iInc].FieldName ).Text ) < eCell^[iInc].Min then
                  eCell^[iInc].Min := StrToFloat(Table.fieldByName( lparrSumFields^[iInc].FieldName ).text);
                 eCell^[iInc].Result := eCell^[iInc].Min;
                end;
               max:                {maximum}
                begin
                 if StrToFloat(Table.fieldByName( lparrSumFields^[iInc].FieldName ).Text ) > eCell^[iInc].Max then
                  eCell^[iInc].Max := StrToFloat(Table.fieldByName( lparrSumFields^[iInc].FieldName ).Text );
                 eCell^[iInc].Result := eCell^[iInc].Max;
                end;
              end;

            end;
          end;

        Table.next;

       end;

      {set var-array to cell-array}
      for iInc := 0 to iSizeOfArray do
       e^[iInc] := eCell^[iInc].Result;

      {call custom method which writes data to cell}
      DrawCellContents(iRow,iCol,e^);

     {clean out var's for next cell}
     for iInc := 0 to iSizeOfArray do
      begin
       eCell^[iInc].Result := 0;   eCell^[iInc].Min := 0;
       eCell^[iInc].Max := 0;      eCell^[iInc].Count := 0;
       eCell^[iInc].Avg := 0;      eCell^[iInc].Sum := 0;
      end;

     end; {end for column FOR..loop}
   end;  {end for row FOR..loop}
   result := true;

 finally
  Dispose(e);
  Dispose(eCell);
 end;

end;


{add summaries to the crosstab. Basically this routine will add a column and row, and
 populate both of these with the sum of the results in the respective cells.}
function TcwXTab.AddSummaries: boolean;
var
 i, iInc, iCol, iRow: integer;
 e, eRet: parrFloat;
begin
 result := true;
 New(e);
 New(eRet);
 try
  {initialise}
  for iInc := 0 to iSizeOfArray do
   begin
    e^[iInc] := 0; eRet^[iInc] := 0;
   end;
  {sum by row}
  for iRow := 1 to (self.rowCount -1) do
   begin
    for iCol := 1 to (self.colCount -1) do
     begin
      {if last column, then add summary}
      if iCol = (self.ColCount -1) then
       begin
        DrawCellContents(iRow,(self.ColCount -1),e^); {draw result in last column}
       end
      else
       begin
        {get cell contents and add to array}
        eRet^ := GetCellContents(iRow,iCol);
        for i := 0 to iSizeOfArray do
         e^[i] := e^[i] + eRet^[i];
       end;
    end;

    {clean up}
    for iInc := 0 to iSizeOfArray do
     begin
      e^[iInc] := 0; eRet^[iInc] := 0;
     end;

   end;


  {sum by column}
  for iCol := 1 to (self.colCount -1) do
   begin
    for iRow := 1 to (self.rowCount -1) do
     begin
      {if last column, then add summary}
      if iRow = (self.RowCount -1) then
       begin
        DrawCellContents((self.RowCount -1),iCol,e^); {draw result in last row}
       end
       else
        begin
         {get cell contents and add to array}
         eRet^ := GetCellContents(iRow,iCol);
         for i := 0 to iSizeOfArray do
          e^[i] := e^[i] + eRet^[i];
        end;
      end;

   {clean up}
   for iInc := 0 to iSizeOfArray do
    begin
     e^[iInc] := 0; eRet^[iInc] := 0;
    end;

  end;
  result := true;

 {cleanup}
 finally
  Dispose(e);
  Dispose(eRet);
 end;


end;


{draw cell contents - takes the array of floats and draws it into the cell as specified in
 the parameters}
procedure TcwXTab.DrawCellContents(const iRow, iCol: integer; eArray: TarrFloat);
var
 sCellNew, sCellCurrent: string;
begin

 {initialise}
 sCellNew := '';
 sCellCurrent := '';

 {iterate through field array - draw to cells the result field in the matching cell array}
 for iInc := 0 to iSizeOfArray do
  begin
   if lparrSumFields^[iInc].FieldName <> '' then
    begin
    {switch on format - notice that currency formats are rounded up}
    if eArray[iInc] <> 0 then
     case lparrSumFields^[iInc].format of
      currency_format:
       sCellNew := format('%0.0m',[eArray[iInc]]);
      integer_format:
       sCellNew := format('%0.0n',[eArray[iInc]]);
      real_format:
       sCellNew := format('%0.2n',[eArray[iInc]]);
     end
    else
     if EmptyCellChar = ecBlank then
      sCellNew := ''
     else
      {switch on format type}
      case lparrSumFields^[iInc].format of
       currency_format:
        sCellNew := '0';
       integer_format:
        sCellNew := '0';
       real_format:
        sCellNew := '0';
      end;


   {we must now write the result to the cell - check to see if we should add a hard-return}
    if iInc < 2 then
     begin
      if lparrSumFields^[iInc + 1].FieldName <> '' then
       sCellCurrent := sCellCurrent + sCellNew +#13#10 {add hard-return to move to next line in cell}
      else
       sCellCurrent := sCellCurrent + sCellNew; {add line, but no hard-return}
     end
    else
     sCellCurrent := sCellCurrent + sCellNew;

   end;
  end;

  {write to cell}
  self.cells[iCol,iRow] := '';                  {empty cell}
  self.cells[iCol,iRow] := sCellCurrent;       {add new value to cell}

end;


{get cell contents - returns the cell contents as an array of float. Note that this
 routine is quite complex, since we have to read the values from the cell and, without
 knowing how many lines there are in the cell, split the text into the array}
function TcwXTab.GetCellContents(const iRow, iCol: integer): TarrFloat;
var
 st1, st2, st3, st4: string;
 e: parrFloat;
begin
 New(e);
 try
  {initialise}
  st1 := ''; st2 := ''; st3 := ''; st4 := '';
  for iInc := 0 to iSizeOfArray do
   e^[iInc] := 0;
  {find cell string}
  st1 := self.cells[iCol,iRow];

  {find individual strings within the cell}
  st2 := copy(st1,1,pos(#13,st1)-1); {copies up to the first line-feed - if there is one}
  if st2 = '' then
   st2 := copy(st1,1,length(st1)) {copies the entire string, in case of no line-feed}
  else
   st3 := copy(st1,pos(#13,st1)+2,255); {definetly a line-feed, so copies from first line-feed onwards}

  st4 := copy(st3,1,pos(#13,st3)-1); {copies up to the next line-feed - if there is one}
  if st4 <> '' then
   begin
    {more than two lines in cell...}
    st4 := copy(st3,pos(#13,st3)+2,255); {copies from the next line-feed onwards}
    st3 := copy(st3,1,pos(#13,st3)-1); {copies up to the next line-feed}
   end;

  {delete currency symbols if they exist - won't work correctly if currencyString greater than
   one character!!!}
  for iInc := 1 to (length(st2)) do
   if st2[iInc] = currencyString then delete(st2,iInc,1);
  for iInc := 1 to (length(st3)) do
   if st3[iInc] = currencyString then delete(st3,iInc,1);
  for iInc := 1 to (length(st4)) do
   if st4[iInc] = currencyString then delete(st4,iInc,1);

  {delete commas if they exist}
  for iInc := 1 to (length(st2)) do
   if st2[iInc] = ',' then delete(st2,iInc,1);
  for iInc := 1 to (length(st3)) do
   if st3[iInc] = ',' then delete(st3,iInc,1);
  for iInc := 1 to (length(st4)) do
   if st4[iInc] = ',' then delete(st4,iInc,1);

  {write values to array (provided not empty)}
  if (Crosstab.SummaryField1 <> '') and not ((st2 = '') or (st2 = #13+#10) or (st2 = #13+#10+#13+#10))
   then e^[0] := StrToFloat(st2);
  if (Crosstab.SummaryField2 <> '') and not ((st3 = '') or (st3 = #13+#10))
   then e^[1] := StrToFloat(st3);
  if (Crosstab.SummaryField3 <> '') and (st4 <>'')
   then e^[2] := StrToFloat(st4);

  result := e^;

 {cleanup}
 finally
  Dispose(e);
 end;


end;


{***Paint Routines**************************************************************}

{draw cell - this routine allows a cell to contain more than one line of text}
procedure TcwXTab.DrawCell(Sender: TObject; Col, Row: Longint; Rect: TRect; State: TGridDrawState);
var
 i, iInc, iRow, iCol : integer;
 lpszStr: pChar;
begin
 lpszStr := StrAlloc(sizeOf(string));
 try
  iRow := Row;
  iCol := Col;

  {exit if cell is column\row - in both these cases we keep default since not aesthetically pleasing
   to have right-aligned text in row and column headers}
  if iCol = 0 then exit;
  if iRow = 0 then exit;

  with (Sender as TStringGrid) do

   with Canvas do
    begin
     {colour summary fields maroon}
     font.color := (Sender as TStringGrid).Font.Color; {...default...}
     if AggRowsCols then
      begin
       if (iCol = (self.colCount -1)) or (iRow = (self.rowCount -1)) then
        font.color := clMaroon
       else
        Font.Color := (Sender as TStringGrid).Font.Color;
      end;

     if (gdSelected in State) then
      begin
       Brush.Color := clHighlight;
       Font.Color  := clHighlightText;
      end
     else
      if (gdFixed in State) then
       Brush.Color := self.FixedColor
      else
       Brush.Color := self.Color;

     FillRect(Rect);
     SetBkMode(Handle, TRANSPARENT);
     StrPCopy(lpszStr,Cells[iCol,iRow]);
     {key line...notice the parameters}
     DrawText(Handle, lpszStr, StrLen(lpszStr), Rect, DT_RIGHT OR DT_WORDBREAK);

    end;

 finally
  StrDispose(lpszStr);
 end;


end;



{***I\O Routines*****************************************************************}

{save crosstab as a table. Note that parameter is the table name, so should be no
 greater than eight characters}
function TcwXTab.SaveXTab(const sTable: string): boolean;
var
 Table1: TTable;
 st1, st2, st3, st4: string;
 i, iRow, iCol: integer;
 e: parrFloat;
 wRow, wCol: word;
begin
 result := false;
 Table1 := TTable.create(self);  New(e);
 try
  {initialise}
  st1 := ''; st2 := ''; st3 := ''; st4 := '';
  for iInc := 0 to iSizeOfArray do
   e^[iInc] := 0;

  {create table}
  with Table1 do
   begin
    Active := False; DatabaseName := Table.DatabaseName;
    TableName := sTable; TableType := ttParadox;
    with FieldDefs do
     begin
      Clear;
      {add fields - note I've made 20 the maximum size of the row\column labels}
      Add('Field1', ftString, 20, false);
      Add('Field2', ftString, 20, false);
      if Crosstab.SummaryField1 <> '' then Add('Field3', ftFloat, 0, false);
      if Crosstab.SummaryField2 <> '' then Add('Field4', ftFloat, 0, false);
      if Crosstab.SummaryField3 <> '' then Add('Field5', ftFloat, 0, false);
     end;
    CreateTable;
    Table1.active := true;
   end;

   {iterate by grid row}
   wRow := self.RowCount -1;
   if AggRowsCols then dec(wRow,1);
   for iRow := 1 to wRow do
    begin

    {iterate by grid column}
    wCol := self.colCount -1;
    if AggRowsCols then dec(wCol,1);
    for iCol := 1 to wCol do
     begin

      {this line fixes a bug which was adding blank rows un-necessary - basically, the FOR loop
       skips this iteration if the column or row value is blank}
      if (self.cells[iCol,0] = '') or (self.cells[0,iRow] = '') then continue;

      {get cell contents}
      e^ := GetCellContents(iRow,iCol);

      {read cell contents into strings}
      st2 := FloatToStr(e^[0]); st3 := FloatToStr(e^[1]); st4 := FloatToStr(e^[2]);

      {add record to table - note column field will be first field in table...}
      if Crosstab.SummaryField3 <> '' then
       table1.AppendRecord([ self.cells[iCol,0],self.cells[0,iRow],StrToFloat(st2),StrToFloat(st3),StrToFloat(st4) ])
      else
       begin
        if Crosstab.SummaryField2 <> '' then
         table1.AppendRecord([ self.cells[iCol,0],self.cells[0,iRow],StrToFloat(st2),StrToFloat(st3) ])
        else
         if Crosstab.SummaryField1 <> '' then table1.AppendRecord([ self.cells[iCol,0],self.cells[0,iRow],StrToFloat(st2) ])
       end;


      {clean-up}
      st1 := ''; st2 := ''; st3 := ''; st4 := '';
      for iInc := 0 to iSizeOfArray do
       e^[iInc] := 0;

     end; {end for column FOR..loop}

    end;  {end for row FOR..loop}

  table1.active := false;

 finally
  Table1.free;
  Dispose(e);
 end;

end;



{}
end.


